/*******************************************************
Filename :  stn_std_datum.sql
Release  :  c2.0.3
SIR	 :  
Description :  Add and Populate std_lat std_long and std_datum fk in fa_station.
Author	 :  Ganesh Thadkamalla
Date	 :  06/10/2004.
Comments :  Oracle Spatial option should be installed on the ETL database
before runing this script.

Following are the rules implemented.

1) Coordinates with WGS72,WGS84 and NAD27 datums are converted to NAD83 datum. 
The converted columns are populated in appropriate STD. columns.
2) For station points other than Point of Record, the
station's Point of Record country/state assignment is used for NAD27 conversion to NAD83.
If Station Point of Record have no state/country assignments, then the coordinate
and datum information is copied to STD columns with out any conversion.
3) Coordinates with any other datums are copied to appropirate STD columns.
____________
Datum conversion srid codes are read from etl_glob_var package.
**********************************************************/
spool stn_std_datum.log

-- Add 3 new columns to fa_station for supporting standard datum.
alter table fa_station
add ( STD_LATITUDE       NUMBER(9,7),
STD_LONGITUDE            NUMBER(10,7),
fk_std_hdatum        NUMBER(12));

set serveroutput on
declare
  cursor cstn is
  select s.pk_isn,s.latitude,s.longitude,
         s.COUNTRY_CODE,s.STATE_POSTAL_CODE, hd.id_code,s.fk_mad_hdatum,
         s.STATION_IS_NUMBER,s.ORGANIZATION_ID,s.LOCATION_POINT_TYPE 
  from fa_station s,
       lu_mad_hdatum hd
  where s.fk_mad_hdatum = hd.pk_isn (+);

  -- geometry objects
  geom1 mdsys.sdo_geometry;
  geom2 mdsys.sdo_geometry;
  --local variables
  lv_lat fa_station.STD_LATITUDE%type;
  lv_long fa_station.STD_LONGITUDE%type;
  lv_std_mad_hdatum fa_station.FK_STD_HDATUM%type;
  lv_nad83 fa_station.FK_STD_HDATUM%type;
  lv_state_postal_code fa_station.state_postal_code%type;
  lv_country_code fa_station.country_code%type;
  lv_cntr number:=0;
  lv_convert number(1) := 0;
  lv_srid number(6,0);


begin
-- Oracle srid for datum that need to be converted.
--WGS72    		8306 
--WGS84    		8307 
--NAD27    		8260
--NAD27 US - Alaska  	8254
--NAD27 cANADA  	8256
--NAD27 MEXICO  	8263
--Puerto Rico ???

  -- fk of nad83
  select pk_isn into lv_nad83
  from lu_mad_hdatum
  where id_code='NAD83';

  for cstn_var in cstn loop
   -- initialize variables
   lv_convert:=0;
   lv_srid:=null;
   -- Determine if conversion is needed. If so, identify the source SRID.
  if cstn_var.latitude is null then 
    lv_convert:= -1;
  elsif cstn_var.id_code  = 'WGS72' then
    lv_srid := etl_glob_var.srid_wgs72;
    lv_convert := 1;
   elsif cstn_var.id_code  = 'WGS84' then
    lv_srid := etl_glob_var.srid_wgs84;
    lv_convert := 1;
   elsif cstn_var.id_code  = 'NAD27' then
     -- fetch POINT OF RECORD country/state information.
      lv_country_code :=null;
      lv_state_postal_code :=null;
    if cstn_var.location_point_type = '*POINT OF RECORD' THEN
      lv_state_postal_code := cstn_var.state_postal_code;
      lv_country_code := cstn_var.country_code;
    else
      BEGIN
      select country_code, state_postal_code 
      into lv_country_code,lv_state_postal_code
      from fa_station fas
      where fas.station_is_number = cstn_var.station_is_number
       and fas.organization_id = cstn_var.organization_id
       and fas.location_point_type = '*POINT OF RECORD';

     EXCEPTION WHEN NO_DATA_FOUND THEN
       LV_COUNTRY_CODE := '00';
       LV_STATE_POSTAL_CODE :='00';
     END;
    end if;

     if lv_country_code='CN' then     
       lv_srid :=etl_glob_var.srid_NAD27_CANADA;
       lv_convert := 1;
     elsif lv_country_code='MX' then     
       lv_srid:=etl_glob_var.srid_NAD27_MEXICO;
       lv_convert := 1;
     elsif lv_country_code='US' and lv_state_postal_code='AK' then   
       lv_srid := etl_glob_var.srid_NAD27_Alaska;
       lv_convert := 1;
     elsif lv_country_code= 'US' then
       -- If not Alaska then it is considered continental US.
       lv_srid:=etl_glob_var.srid_NAD27;
       lv_convert := 1;
     else 
       lv_convert := 0;
     end if;  
   else
    lv_convert:=0;
   end if;


   if lv_convert = 1 then
    -- initialize/reset variables.
    geom1:=null;
    geom2:=null;
    lv_lat :=null;
    lv_long :=null;

    lv_std_mad_hdatum := lv_nad83;
    geom1:= mdsys.sdo_geometry(2001,lv_srid,
             mdsys.sdo_point_type(cstn_var.longitude,
                   cstn_var.latitude,null),
                 null,null);
    geom2:=mdsys.sdo_cs.transform(geom1, to_srid=>etl_glob_var.srid_NAD83); --convert to nad 83
    --pull the x/y back out
    lv_long:=geom2.sdo_point.x;
    lv_lat:=geom2.sdo_point.y;

    -- update 
    update fa_station c
    set c.std_latitude = lv_lat,
        c.std_longitude = lv_long,
        c.fk_std_hdatum = lv_std_mad_hdatum
    where c.pk_isn = cstn_var.pk_isn;

   elsif lv_convert = 0 then
    update fa_station c
    set c.std_latitude = cstn_var.latitude,
        c.std_longitude = cstn_var.longitude,
        c.fk_std_hdatum = cstn_var.fk_mad_hdatum
    where c.pk_isn = cstn_var.pk_isn;
   elsif lv_convert = -1 then
    null;
   end if;
    -- perform intermediate commits.
    lv_cntr:=lv_cntr+1;
    if lv_cntr > 3000 then
      commit;
      lv_cntr:=0;
    end if;

  end loop;

  commit;
end;
/
show errors

--Creating fk constraint and index on fk_std_hdatum
alter table fa_station
add constraint fk_sd_std_mad_hdatum foreign key (fk_std_hdatum ) references lu_mad_hdatum(pk_isn);

-- indexes
create index isd_std_mad_hdatum on fa_station(fk_std_hdatum)
tablespace STORETINDX
storage (initial 5M next 3M)
nologging;

create index sd_std_lat on                             
fa_station(std_latitude)                          
tablespace storetindx                             
storage(initial 3m next 3m) nologging;                       
                                                   
create index sd_std_long on                            
fa_station(std_longitude)                         
tablespace storetindx                             
storage(initial 3m next 3m) nologging;   

spool off;